<?php

namespace App\Http\Controllers;

use App\Models\Page;
use App\Models\Video;
use App\Models\VideoMetric;
use App\Models\AudienceSegment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use Carbon\Carbon;

class VideoController extends Controller
{
    /**
     * Exibe a lista de vídeos.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $query = Video::query()
            ->join('pages', 'videos.page_id', '=', 'pages.id')
            ->where('pages.user_id', Auth::id())
            ->select('videos.*');
            
        // Filtros
        if ($request->has('page_id')) {
            $query->where('page_id', $request->page_id);
        }
        
        if ($request->has('video_type')) {
            $query->where('video_type', $request->video_type);
        }
        
        if ($request->has('date_from')) {
            $query->where('published_at', '>=', Carbon::parse($request->date_from));
        }
        
        if ($request->has('date_to')) {
            $query->where('published_at', '<=', Carbon::parse($request->date_to));
        }
        
        if ($request->has('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('title', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }
        
        $videos = $query->orderBy('published_at', 'desc')->paginate(15);
        $pages = Page::where('user_id', Auth::id())->get();
        
        return view('videos.index', compact('videos', 'pages'));
    }

    /**
     * Importa vídeos de uma página específica do Facebook.
     *
     * @param  int  $pageId
     * @return \Illuminate\Http\Response
     */
    public function import($pageId)
    {
        try {
            $page = Page::where('user_id', Auth::id())->findOrFail($pageId);
            
            // Verificar se o token de acesso da página está disponível
            if (empty($page->access_token)) {
                return redirect()->route('pages.show', $page->id)
                    ->with('error', 'Token de acesso da página não encontrado. Por favor, reimporte a página.');
            }
            
            // Fazer requisição à API do Facebook para obter os vídeos
            $response = Http::get("https://graph.facebook.com/v18.0/{$page->facebook_page_id}/videos", [
                'access_token' => $page->access_token,
                'fields' => 'id,title,description,picture,permalink_url,length,created_time,status,views',
                'limit' => 50
            ]);
            
            if (!$response->successful()) {
                return redirect()->route('pages.show', $page->id)
                    ->with('error', 'Erro ao importar vídeos: ' . $response->json('error.message', 'Erro desconhecido'));
            }
            
            $videosData = $response->json('data', []);
            $importCount = 0;
            
            foreach ($videosData as $videoData) {
                // Determinar o tipo de vídeo (regular ou reel)
                $videoType = 'regular'; // Por padrão, consideramos como vídeo regular
                
                // Atualizar ou criar o vídeo no banco de dados
                $video = Video::updateOrCreate(
                    [
                        'page_id' => $page->id,
                        'facebook_video_id' => $videoData['id']
                    ],
                    [
                        'title' => $videoData['title'] ?? '',
                        'description' => $videoData['description'] ?? '',
                        'thumbnail_url' => $videoData['picture'] ?? null,
                        'permalink_url' => $videoData['permalink_url'] ?? null,
                        'duration' => $videoData['length'] ?? 0,
                        'published_at' => Carbon::parse($videoData['created_time']),
                        'video_type' => $videoType,
                        'status' => $videoData['status']['video_status'] ?? 'active',
                    ]
                );
                
                // Importar métricas básicas
                if (isset($videoData['views'])) {
                    VideoMetric::updateOrCreate(
                        [
                            'video_id' => $video->id,
                            'metric_name' => 'views',
                            'period' => 'lifetime'
                        ],
                        [
                            'metric_value' => (string) $videoData['views'],
                            'date_start' => null,
                            'date_end' => Carbon::now(),
                        ]
                    );
                }
                
                $importCount++;
            }
            
            // Importar métricas detalhadas e dados de público para cada vídeo
            foreach (Video::where('page_id', $page->id)->get() as $video) {
                $this->importVideoMetrics($video, $page->access_token);
                $this->importAudienceData($video, $page->access_token);
            }
            
            return redirect()->route('pages.show', $page->id)
                ->with('success', "Importação concluída! {$importCount} vídeos foram importados com sucesso.");
            
        } catch (\Exception $e) {
            return redirect()->route('pages.index')
                ->with('error', 'Erro ao importar vídeos: ' . $e->getMessage());
        }
    }

    /**
     * Importa métricas detalhadas para um vídeo específico.
     *
     * @param  Video  $video
     * @param  string  $accessToken
     * @return void
     */
    private function importVideoMetrics(Video $video, $accessToken)
    {
        try {
            $response = Http::get("https://graph.facebook.com/v18.0/{$video->facebook_video_id}/video_insights", [
                'access_token' => $accessToken,
                'metric' => 'total_video_views,total_video_views_unique,total_video_complete_views,total_video_complete_views_unique,total_video_30s_views,total_video_30s_views_unique,total_video_avg_time_watched,total_video_view_total_time,total_video_impressions,total_video_impressions_unique,total_video_reactions_by_type_total'
            ]);
            
            if (!$response->successful()) {
                return;
            }
            
            $metricsData = $response->json('data', []);
            
            foreach ($metricsData as $metricData) {
                $metricName = $metricData['name'];
                $metricValue = is_array($metricData['values'][0]['value']) 
                    ? json_encode($metricData['values'][0]['value'])
                    : (string) $metricData['values'][0]['value'];
                
                VideoMetric::updateOrCreate(
                    [
                        'video_id' => $video->id,
                        'metric_name' => $metricName,
                        'period' => 'lifetime'
                    ],
                    [
                        'metric_value' => $metricValue,
                        'date_start' => null,
                        'date_end' => Carbon::now(),
                    ]
                );
            }
        } catch (\Exception $e) {
            // Log do erro, mas continua a execução
            \Log::error("Erro ao importar métricas para o vídeo {$video->id}: " . $e->getMessage());
        }
    }

    /**
     * Importa dados de público para um vídeo específico.
     *
     * @param  Video  $video
     * @param  string  $accessToken
     * @return void
     */
    private function importAudienceData(Video $video, $accessToken)
    {
        try {
            // Importar dados demográficos (idade e gênero)
            $response = Http::get("https://graph.facebook.com/v18.0/{$video->facebook_video_id}/video_insights", [
                'access_token' => $accessToken,
                'metric' => 'total_video_view_time_by_age_bucket_and_gender'
            ]);
            
            if (!$response->successful()) {
                return;
            }
            
            $demographicData = $response->json('data.0.values.0.value', []);
            
            foreach ($demographicData as $gender => $ageGroups) {
                foreach ($ageGroups as $ageGroup => $viewTime) {
                    // Calcular taxa de engajamento (simplificada)
                    $viewCount = $viewTime > 0 ? ceil($viewTime / 10) : 0; // Estimativa simplificada
                    $engagementRate = $viewTime > 0 ? min(100, $viewTime / 100) : 0; // Valor arbitrário para exemplo
                    
                    AudienceSegment::updateOrCreate(
                        [
                            'video_id' => $video->id,
                            'segment_type' => 'demographic',
                            'segment_value' => "{$gender}_{$ageGroup}"
                        ],
                        [
                            'view_count' => $viewCount,
                            'engagement_rate' => $engagementRate
                        ]
                    );
                }
            }
            
            // Importar dados de localização
            $response = Http::get("https://graph.facebook.com/v18.0/{$video->facebook_video_id}/video_insights", [
                'access_token' => $accessToken,
                'metric' => 'total_video_view_time_by_country_id'
            ]);
            
            if (!$response->successful()) {
                return;
            }
            
            $locationData = $response->json('data.0.values.0.value', []);
            
            foreach ($locationData as $countryCode => $viewTime) {
                // Calcular taxa de engajamento (simplificada)
                $viewCount = $viewTime > 0 ? ceil($viewTime / 10) : 0; // Estimativa simplificada
                $engagementRate = $viewTime > 0 ? min(100, $viewTime / 100) : 0; // Valor arbitrário para exemplo
                
                AudienceSegment::updateOrCreate(
                    [
                        'video_id' => $video->id,
                        'segment_type' => 'location',
                        'segment_value' => $countryCode
                    ],
                    [
                        'view_count' => $viewCount,
                        'engagement_rate' => $engagementRate
                    ]
                );
            }
        } catch (\Exception $e) {
            // Log do erro, mas continua a execução
            \Log::error("Erro ao importar dados de público para o vídeo {$video->id}: " . $e->getMessage());
        }
    }

    /**
     * Exibe os detalhes de um vídeo específico.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $video = Video::findOrFail($id);
        
        // Verificar se o vídeo pertence a uma página do usuário atual
        $page = Page::where('id', $video->page_id)
            ->where('user_id', Auth::id())
            ->firstOrFail();
        
        $metrics = $video->metrics()->get();
        $audienceSegments = $video->audienceSegments()->get();
        
        return view('videos.show', compact('video', 'page', 'metrics', 'audienceSegments'));
    }

    /**
     * Atualiza as métricas de um vídeo específico.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function updateMetrics($id)
    {
        $video = Video::findOrFail($id);
        
        // Verificar se o vídeo pertence a uma página do usuário atual
        $page = Page::where('id', $video->page_id)
            ->where('user_id', Auth::id())
            ->firstOrFail();
        
        try {
            $this->importVideoMetrics($video, $page->access_token);
            $this->importAudienceData($video, $page->access_token);
            
            return redirect()->route('videos.show', $video->id)
                ->with('success', 'Métricas atualizadas com sucesso!');
        } catch (\Exception $e) {
            return redirect()->route('videos.show', $video->id)
                ->with('error', 'Erro ao atualizar métricas: ' . $e->getMessage());
        }
    }
}
