<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

/**
 * Modelo para representar um anúncio de vídeo do Facebook
 */
class VideoAd extends Model
{
    /**
     * A tabela associada ao modelo.
     *
     * @var string
     */
    protected $table = 'video_ads';

    /**
     * Os atributos que são atribuíveis em massa.
     *
     * @var array
     */
    protected $fillable = [
        'facebook_ad_id',
        'ad_creative_body',
        'ad_creative_link_title',
        'ad_snapshot_url',
        'page_name',
        'publisher_platforms',
        'impressions_lower',
        'impressions_upper',
        'spend_lower',
        'spend_upper',
        'currency',
        'demographic_distribution',
        'video_url',
        'category',
        'views',
        'shares',
        'conversions',
        'engagement_data',
        'audience_insights',
        'performance_metrics',
    ];

    /**
     * Os atributos que devem ser convertidos para tipos nativos.
     *
     * @var array
     */
    protected $casts = [
        'publisher_platforms' => 'array',
        'demographic_distribution' => 'array',
        'engagement_data' => 'array',
        'audience_insights' => 'array',
        'performance_metrics' => 'array',
    ];

    /**
     * Escopo para filtrar anúncios por categoria
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $category
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeCategory($query, $category)
    {
        return $query->where('category', $category);
    }

    /**
     * Escopo para filtrar anúncios por número mínimo de visualizações
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $minViews
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeMinViews($query, $minViews)
    {
        return $query->where('views', '>=', $minViews);
    }

    /**
     * Escopo para filtrar anúncios por número mínimo de conversões
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $minConversions
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeMinConversions($query, $minConversions)
    {
        return $query->where('conversions', '>=', $minConversions);
    }

    /**
     * Escopo para filtrar anúncios por número mínimo de compartilhamentos
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param int $minShares
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeMinShares($query, $minShares)
    {
        return $query->where('shares', '>=', $minShares);
    }

    /**
     * Escopo para buscar anúncios por texto
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string $searchTerm
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearch($query, $searchTerm)
    {
        return $query->where(function ($query) use ($searchTerm) {
            $query->where('ad_creative_body', 'LIKE', "%{$searchTerm}%")
                  ->orWhere('ad_creative_link_title', 'LIKE', "%{$searchTerm}%")
                  ->orWhere('page_name', 'LIKE', "%{$searchTerm}%");
        });
    }
}
