<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Services\TopicCategorizer;

/**
 * Serviço para análise de vídeos de terceiros
 */
class ThirdPartyVideoAnalyzer
{
    /**
     * Serviço de categorização de tópicos
     *
     * @var TopicCategorizer
     */
    protected $topicCategorizer;

    /**
     * Construtor
     *
     * @param TopicCategorizer $topicCategorizer
     */
    public function __construct(TopicCategorizer $topicCategorizer)
    {
        $this->topicCategorizer = $topicCategorizer;
    }

    /**
     * Analisa vídeos de terceiros com base em critérios específicos
     *
     * @param array $searchParams
     * @return array
     */
    public function analyzeThirdPartyVideos($searchParams)
    {
        // Em um ambiente real, aqui faríamos a chamada para a Ad Library API
        // Como estamos usando dados simulados, vamos gerar resultados de análise
        
        $videos = $this->getThirdPartyVideos($searchParams);
        
        // Categorizar vídeos
        foreach ($videos as &$video) {
            $categorization = $this->topicCategorizer->categorizeVideo($video);
            $video['category'] = $categorization['main_category'];
            $video['sub_categories'] = $categorization['sub_categories'];
        }
        
        // Análise de temas populares
        $themeAnalysis = $this->topicCategorizer->analyzePopularThemes($videos);
        
        // Análise de desempenho por categoria
        $performanceAnalysis = $this->topicCategorizer->analyzePerformanceByCategory($videos);
        
        // Análise de público-alvo
        $audienceAnalysis = $this->analyzeAudienceByCategory($videos);
        
        return [
            'videos' => $videos,
            'theme_analysis' => $themeAnalysis,
            'performance_analysis' => $performanceAnalysis,
            'audience_analysis' => $audienceAnalysis
        ];
    }

    /**
     * Obtém vídeos de terceiros com base em parâmetros de pesquisa
     *
     * @param array $searchParams
     * @return array
     */
    protected function getThirdPartyVideos($searchParams)
    {
        // Em um ambiente real, aqui faríamos a chamada para a Ad Library API
        // Como estamos usando dados simulados, vamos retornar dados de exemplo
        
        $category = $searchParams['category'] ?? null;
        $searchTerms = $searchParams['search_terms'] ?? '';
        
        // Dados simulados de vídeos
        $videos = $this->generateSampleVideos();
        
        // Filtrar por categoria se especificada
        if ($category) {
            $videos = array_filter($videos, function($video) use ($category) {
                return $video['category'] === $category;
            });
        }
        
        // Filtrar por termos de pesquisa se especificados
        if ($searchTerms) {
            $videos = array_filter($videos, function($video) use ($searchTerms) {
                return stripos($video['ad_creative_body'], $searchTerms) !== false || 
                       stripos($video['ad_creative_link_title'], $searchTerms) !== false;
            });
        }
        
        return array_values($videos);
    }

    /**
     * Gera dados de exemplo para vídeos
     *
     * @return array
     */
    protected function generateSampleVideos()
    {
        return [
            [
                'id' => 'ad_123456789',
                'ad_creative_body' => 'Descubra nossa nova coleção de roupas femininas para o verão! Peças leves, coloridas e com descontos imperdíveis. Compre agora e receba em casa com frete grátis.',
                'ad_creative_link_title' => 'Nova Coleção Verão - Roupas Femininas com 30% OFF',
                'ad_snapshot_url' => 'https://www.facebook.com/ads/library/?id=123456789',
                'page_name' => 'Moda Verão Store',
                'publisher_platforms' => ['facebook', 'instagram'],
                'impressions_lower' => 50000,
                'impressions_upper' => 100000,
                'spend_lower' => 1000,
                'spend_upper' => 5000,
                'currency' => 'BRL',
                'video_url' => 'https://example.com/videos/123456789',
                'category' => 'Moda',
                'views' => 75000,
                'shares' => 3200,
                'conversions' => 1850,
                'demographic_distribution' => [
                    ['age' => '18-24', 'gender' => 'female', 'percentage' => 35],
                    ['age' => '25-34', 'gender' => 'female', 'percentage' => 40],
                    ['age' => '35-44', 'gender' => 'female', 'percentage' => 15],
                    ['age' => '18-24', 'gender' => 'male', 'percentage' => 3],
                    ['age' => '25-34', 'gender' => 'male', 'percentage' => 5],
                    ['age' => '35-44', 'gender' => 'male', 'percentage' => 2]
                ],
                'audience_insights' => [
                    'regions' => [
                        ['region' => 'São Paulo', 'percentage' => 45],
                        ['region' => 'Rio de Janeiro', 'percentage' => 25],
                        ['region' => 'Minas Gerais', 'percentage' => 15],
                        ['region' => 'Bahia', 'percentage' => 10],
                        ['region' => 'Outros', 'percentage' => 5]
                    ],
                    'interests' => [
                        ['interest' => 'Moda', 'percentage' => 85],
                        ['interest' => 'Compras online', 'percentage' => 75],
                        ['interest' => 'Tendências de moda', 'percentage' => 65],
                        ['interest' => 'Estilo de vida', 'percentage' => 55],
                        ['interest' => 'Beleza', 'percentage' => 45]
                    ]
                ],
                'engagement' => [
                    'reactions' => 12500,
                    'comments' => 3800,
                    'shares' => 3200,
                    'clicks' => 25000
                ],
                'performance_metrics' => [
                    'ctr' => 3.2,
                    'cpc' => 0.85,
                    'cpm' => 45.50,
                    'conversion_rate' => 2.5,
                    'roi' => 3.8,
                    'completion_rate' => 65,
                    'trend' => 15,
                    'peak_date' => '15/03/2025',
                    'peak_views' => 8500,
                    'peak_conversions' => 320,
                    'sector_comparison' => 25
                ]
            ],
            [
                'id' => 'ad_987654321',
                'ad_creative_body' => 'Revolucione sua cozinha com o novo Processador Multifuncional Pro! Corta, pica, rala, mistura e muito mais. Com 10 velocidades e design moderno, é o aparelho que vai transformar suas receitas. Aproveite o desconto especial!',
                'ad_creative_link_title' => 'Processador Multifuncional Pro - 40% OFF por tempo limitado',
                'ad_snapshot_url' => 'https://www.facebook.com/ads/library/?id=987654321',
                'page_name' => 'Eletro Casa',
                'publisher_platforms' => ['facebook', 'instagram', 'messenger'],
                'impressions_lower' => 30000,
                'impressions_upper' => 80000,
                'spend_lower' => 800,
                'spend_upper' => 3000,
                'currency' => 'BRL',
                'video_url' => 'https://example.com/videos/987654321',
                'category' => 'Eletrônicos',
                'views' => 55000,
                'shares' => 2100,
                'conversions' => 1200,
                'demographic_distribution' => [
                    ['age' => '25-34', 'gender' => 'female', 'percentage' => 30],
                    ['age' => '35-44', 'gender' => 'female', 'percentage' => 35],
                    ['age' => '45-54', 'gender' => 'female', 'percentage' => 15],
                    ['age' => '25-34', 'gender' => 'male', 'percentage' => 8],
                    ['age' => '35-44', 'gender' => 'male', 'percentage' => 7],
                    ['age' => '45-54', 'gender' => 'male', 'percentage' => 5]
                ],
                'audience_insights' => [
                    'regions' => [
                        ['region' => 'São Paulo', 'percentage' => 40],
                        ['region' => 'Rio de Janeiro', 'percentage' => 20],
                        ['region' => 'Minas Gerais', 'percentage' => 15],
                        ['region' => 'Rio Grande do Sul', 'percentage' => 15],
                        ['region' => 'Outros', 'percentage' => 10]
                    ],
                    'interests' => [
                        ['interest' => 'Culinária', 'percentage' => 80],
                        ['interest' => 'Eletrodomésticos', 'percentage' => 75],
                        ['interest' => 'Decoração', 'percentage' => 60],
                        ['interest' => 'Gastronomia', 'percentage' => 55],
                        ['interest' => 'Compras online', 'percentage' => 50]
                    ]
                ],
                'engagement' => [
                    'reactions' => 8500,
                    'comments' => 2200,
                    'shares' => 2100,
                    'clicks' => 18000
                ],
                'performance_metrics' => [
                    'ctr' => 2.8,
                    'cpc' => 1.05,
                    'cpm' => 38.50,
                    'conversion_rate' => 2.2,
                    'roi' => 3.2,
                    'completion_rate' => 58,
                    'trend' => 8,
                    'peak_date' => '22/03/2025',
                    'peak_views' => 6200,
                    'peak_conversions' => 180,
                    'sector_comparison' => 15
                ]
            ],
            [
                'id' => 'ad_456789123',
                'ad_creative_body' => 'Seu smartphone merece o melhor! Conheça o novo XPhone Pro com câmera de 108MP, tela AMOLED de 6.7" e bateria que dura o dia todo. Disponível em 4 cores incríveis. Parcele em até 12x sem juros!',
                'ad_creative_link_title' => 'Novo XPhone Pro - Tecnologia de ponta com preço especial',
                'ad_snapshot_url' => 'https://www.facebook.com/ads/library/?id=456789123',
                'page_name' => 'TechStore Brasil',
                'publisher_platforms' => ['facebook', 'instagram', 'audience_network'],
                'impressions_lower' => 100000,
                'impressions_upper' => 200000,
                'spend_lower' => 5000,
                'spend_upper' => 10000,
                'currency' => 'BRL',
                'video_url' => 'https://example.com/videos/456789123',
                'category' => 'Eletrônicos',
                'views' => 150000,
                'shares' => 8500,
                'conversions' => 3200,
                'demographic_distribution' => [
                    ['age' => '18-24', 'gender' => 'male', 'percentage' => 25],
                    ['age' => '25-34', 'gender' => 'male', 'percentage' => 30],
                    ['age' => '35-44', 'gender' => 'male', 'percentage' => 15],
                    ['age' => '18-24', 'gender' => 'female', 'percentage' => 15],
                    ['age' => '25-34', 'gender' => 'female', 'percentage' => 10],
                    ['age' => '35-44', 'gender' => 'female', 'percentage' => 5]
                ],
                'audience_insights' => [
                    'regions' => [
                        ['region' => 'São Paulo', 'percentage' => 35],
                        ['region' => 'Rio de Janeiro', 'percentage' => 20],
                        ['region' => 'Minas Gerais', 'percentage' => 15],
                        ['region' => 'Paraná', 'percentage' => 15],
                        ['region' => 'Outros', 'percentage' => 15]
                    ],
                    'interests' => [
                        ['interest' => 'Tecnologia', 'percentage' => 90],
                        ['interest' => 'Smartphones', 'percentage' => 85],
                        ['interest' => 'Gadgets', 'percentage' => 70],
                        ['interest' => 'Jogos', 'percentage' => 65],
                        ['interest' => 'Fotografia', 'percentage' => 60]
                    ]
                ],
                'engagement' => [
                    'reactions' => 25000,
                    'comments' => 7800,
                    'shares' => 8500,
                    'clicks' => 45000
                ],
                'performance_metrics' => [
                    'ctr' => 3.5,
                    'cpc' => 1.25,
                    'cpm' => 52.50,
                    'conversion_rate' => 2.1,
                    'roi' => 4.2,
                    'completion_rate' => 72,
                    'trend' => 22,
                    'peak_date' => '10/03/2025',
                    'peak_views' => 18500,
                    'peak_conversions' => 420,
                    'sector_comparison' => 35
                ]
            ],
            [
                'id' => 'ad_654321987',
                'ad_creative_body' => 'Cuide da sua saúde com os melhores suplementos! Vitamina C, Zinco e Vitamina D em um único comprimido. Fortaleça seu sistema imunológico e mantenha-se saudável. Compre agora e ganhe 15% de desconto na primeira compra!',
                'ad_creative_link_title' => 'Suplemento Imuno Plus - Sua saúde em primeiro lugar',
                'ad_snapshot_url' => 'https://www.facebook.com/ads/library/?id=654321987',
                'page_name' => 'Vida Saudável Suplementos',
                'publisher_platforms' => ['facebook', 'instagram'],
                'impressions_lower' => 40000,
                'impressions_upper' => 90000,
                'spend_lower' => 1200,
                'spend_upper' => 4000,
                'currency' => 'BRL',
                'video_url' => 'https://example.com/videos/654321987',
                'category' => 'Saúde e Bem-estar',
                'views' => 65000,
                'shares' => 4200,
                'conversions' => 2100,
                'demographic_distribution' => [
                    ['age' => '25-34', 'gender' => 'female', 'percentage' => 25],
                    ['age' => '35-44', 'gender' => 'female', 'percentage' => 30],
                    ['age' => '45-54', 'gender' => 'female', 'percentage' => 20],
                    ['age' => '25-34', 'gender' => 'male', 'percentage' => 10],
                    ['age' => '35-44', 'gender' => 'male', 'percentage' => 10],
                    ['age' => '45-54', 'gender' => 'male', 'percentage' => 5]
                ],
                'audience_insights' => [
                    'regions' => [
                        ['region' => 'São Paulo', 'percentage' => 30],
                        ['region' => 'Rio de Janeiro', 'percentage' => 25],
                        ['region' => 'Minas Gerais', 'percentage' => 15],
                        ['region' => 'Santa Catarina', 'percentage' => 15],
                        ['region' => 'Outros', 'percentage' => 15]
                    ],
                    'interests' => [
                        ['interest' => 'Saúde', 'percentage' => 85],
                        ['interest' => 'Bem-estar', 'percentage' => 80],
                        ['interest' => 'Fitness', 'percentage' => 65],
                        ['interest' => 'Nutrição', 'percentage' => 60],
                        ['interest' => 'Vida saudável', 'percentage' => 55]
                    ]
                ],
                'engagement' => [
                    'reactions' => 9500,
                    'comments' => 3200,
                    'shares' => 4200,
                    'clicks' => 22000
                ],
                'performance_metrics' => [
                    'ctr' => 3.0,
                    'cpc' => 0.95,
                    'cpm' => 42.50,
                    'conversion_rate' => 
(Content truncated due to size limit. Use line ranges to read in chunks)